<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class FileManagerController extends Controller
{
    private $root;
    private $tempDir;
    private $backupDir;

    public function __construct()
    {
        // 📂 Dossier parent de base_path()
        $this->root = dirname(dirname(base_path()));

        $this->tempDir = storage_path('temp/code');
        $this->backupDir = storage_path('backups/code');

        // Crée les répertoires nécessaires
        if (!File::exists($this->tempDir)) File::makeDirectory($this->tempDir, 0755, true);
        if (!File::exists($this->backupDir)) File::makeDirectory($this->backupDir, 0755, true);
    }

    // 🔒 Sécurise le chemin pour rester dans le root parent
    private function securePath($relative)
    {
        $path = realpath($this->root . '/' . $relative);

        if (!$path || strpos($path, $this->root) !== 0) {
            abort(403, 'Accès non autorisé');
        }

        // Empêche l'accès aux fichiers sensibles
        if (preg_match('/(\.envv|vendor|storagee|node_modules)/', $path)) {
            abort(403, 'Fichier protégé');
        }

        return $path;
    }

    public function index()
    {
        return view('file-manager');
    }

    // 📂 Liste des fichiers/dossiers
    public function listFiles(Request $request)
    {
        $dir = $request->get('dir', '');
        $path = $this->securePath($dir);

        if (!is_dir($path)) {
            return response()->json(['error' => 'Invalid directory'], 400);
        }

        $dirs = [];
        $files = [];

        foreach (scandir($path) as $item) {
            if ($item === '.' || $item === '..') continue;

            $fullPath = $path . DIRECTORY_SEPARATOR . $item;
            $relativePath = ltrim($dir . '/' . $item, '/');
            $depth = substr_count($relativePath, '/');

            $entry = [
                'name' => $item,
                'path' => $relativePath,
                'type' => is_dir($fullPath) ? 'dir' : 'file',
                'depth' => $depth,
            ];

            if (is_dir($fullPath)) $dirs[] = $entry;
            else $files[] = $entry;
        }

        return response()->json(array_merge($dirs, $files));
    }

    // 📄 Ouvrir un fichier (priorité au temp)
    public function openFile(Request $request)
    {
        $relative = $request->get('path');
        $path = $this->securePath($relative);
        $tempPath = $this->tempDir . '/' . str_replace(['/', '\\'], '_', $relative) . '.tmp';

        if (!File::exists($path)) {
            return response()->json(['error' => 'File not found'], 404);
        }

        if (File::exists($tempPath)) {
            return response()->json([
                'content' => File::get($tempPath),
                'fromTemp' => true,
                'message' => '⚠️ Fichier temporaire détecté (non sauvegardé)'
            ]);
        }

        return response()->json([
            'content' => File::get($path),
            'fromTemp' => false
        ]);
    }

    // 🔁 Auto-save
    public function autoSave(Request $request)
    {
        $relative = $request->get('path');
        $content = $request->get('content');

        if (!$relative || $content === null) {
            return response()->json(['error' => 'Paramètres manquants'], 400);
        }

        $tempPath = $this->tempDir . '/' . str_replace(['/', '\\'], '_', $relative) . '.tmp';
        File::put($tempPath, $content);

        return response()->json(['success' => true,'tempPath' => $tempPath]);
    }

    // 💾 Sauvegarder un fichier
    public function saveFile(Request $request)
    {
        $relative = $request->get('path');
        $content = $request->get('content');

        $path = $this->securePath($relative);
        $tempPath = $this->tempDir . '/' . str_replace(['/', '\\'], '_', $relative) . '.tmp';

        if (!File::exists($path)) {
            return response()->json(['error' => 'File not found'], 404);
        }

        // Backup
        $backupName = str_replace(['/', '\\'], '_', $relative) . '_' . date('Ymd_His') . '.bak';
        $backupPath = $this->backupDir . '/' . $backupName;
        File::copy($path, $backupPath);

        // Écriture
        File::put($path, $content);

        $existe = File::exists($tempPath) ? 'oui' : 'non';

        if (File::exists($tempPath)) {
            File::delete($tempPath);
        }

        return response()->json([
            'success' => true,
            'backup' => $backupName,
            'tempPath' => $tempPath,
            'existe' => $existe
        ]);
    }

    // ➕ Créer fichier ou dossier
    public function createFileOrDir(Request $request)
    {
        $base = $this->securePath($request->get('path', ''));
        $name = $request->get('name');
        $type = $request->get('type');

        if (!$name || !$type) {
            return response()->json(['success' => false, 'message' => 'Nom ou type manquant']);
        }

        $target = $base . DIRECTORY_SEPARATOR . $name;

        if (File::exists($target)) {
            return response()->json(['success' => false, 'message' => 'Déjà existant']);
        }

        if ($type === 'file') {
            File::put($target, "<?php\n\n// Nouveau fichier créé le " . now() . "\n");
        } else {
            File::makeDirectory($target);
        }

        return response()->json(['success' => true]);
    }

    // ✏️ Renommer
    public function renameFile(Request $request)
    {
        $old = $this->securePath($request->get('old'));
        $new = $this->securePath($request->get('new'));

        if (!File::exists($old)) return response()->json(['success' => false]);

        rename($old, $new);
        return response()->json(['success' => true]);
    }

    // 🗑️ Supprimer
    public function deleteFile(Request $request)
    { 
        $path = $this->securePath($request->get('path'));
        if (!File::exists($path)) return response()->json(['success' => false]);

        if (is_dir($path)) File::deleteDirectory($path);
        else File::delete($path);

        return response()->json(['success' => true]);
    }

    // 📤 Upload fichier
    public function upload(Request $request)
    { 
        $request->validate([
            'file' => 'required|file',
            'targetDir' => 'nullable|string',
        ]);

        $file = $request->file('file');
        $targetDir = $request->input('targetDir', '');
        $destination = $this->securePath($targetDir);

        if (!file_exists($destination)) {
            mkdir($destination, 0755, true);
        }

        $filename = $file->getClientOriginalName();
        $file->move($destination, $filename);

        return response()->json([
            'success' => true,
            'path' => trim($targetDir . '/' . $filename, '/')
        ]);
    }
}