@extends('layouts.app')

@section('content')
<style>
ul#file-tree{
  height: 100vh;
}
</style>
<div class="container-fluid">
    <div class="row" style="height:90vh;">
        <!-- 🌳 Arborescence -->
        <div class="col-md-3 bg-dark text-light p-2" style="overflow:auto;">
            <div class="d-flex justify-content-between align-items-center mb-2">
                <h5 class="text-center mb-0">📁 Explorateur</h5>
                <div>
                    <button class="btn btn-sm btn-outline-light" onclick="newItem('file')">📄</button>
                    <button class="btn btn-sm btn-outline-light" onclick="newItem('dir')">📁</button>
                </div>
            </div>
            <div id="file-tree" style="height: 100vh;" class="list-unstyled"></div>
        </div>

        <!-- 💻 Éditeur -->
        <div class="col-md-9 p-3 bg-light">
            <div class="d-flex justify-content-between align-items-center">
                <span id="file-path" class="text-muted small"></span>
                <button id="save-btn" class="btn btn-success btn-sm">💾 EnregistrerR</button>
            </div>
            <div class="mb-2">
                <input type="file" id="file-upload" class="form-control-sm" />
                <button id="upload-btn" class="btn btn-primary btn-sm">⬆️ Upload</button>
            </div>

            <div class="d-flex align-items-center border-bottom mb-1" id="tabs-bar" style="overflow-x:auto; white-space:nowrap;"></div>

            <div id="editor" style="height:100vh; border:1px solid #ccc; margin-top:5px;"></div>
        </div>
    </div>
</div>

<!-- Ace Editor chadoss kas FFF AUTO-->
<script src="https://cdnjs.cloudflare.com/ajax/libs/ace/1.32.3/ace.js"></script>

<script>

let currentPath = '';
let loadedDirs = {};
let editor;
let autoSaveInterval = null;
const openedDirs = new Set();

let openFiles = {}; // clé = path, valeur = { session, tabEl }
let activeFile = null;

function getModeByExtension(file) {
    if (file.endsWith('.php')) return 'ace/mode/php';
    if (file.endsWith('.js')) return 'ace/mode/javascript';
    if (file.endsWith('.css')) return 'ace/mode/css';
    if (file.endsWith('.html') || file.endsWith('.blade.php')) return 'ace/mode/html';
    if (file.endsWith('.json')) return 'ace/mode/json';
    if (file.endsWith('.md')) return 'ace/mode/markdown';
    return 'ace/mode/text';
}

function initEditor() {
    if (!editor) {
        editor = ace.edit("editor");
        editor.setTheme("ace/theme/dracula");
        editor.setFontSize(14);
        editor.session.setTabSize(4);
        editor.session.setUseSoftTabs(true);
    }
    return editor;
}

function createTab(path) {
    const tabsBar = document.getElementById('tabs-bar');
    const fileName = path.split('/').pop();

    const tab = document.createElement('div');
    tab.className = 'tab-item px-2 py-1 border-end';
    tab.style.cursor = 'pointer';
    tab.style.display = 'inline-flex';
    tab.style.alignItems = 'center';
    tab.style.background = '#eee';
    tab.style.userSelect = 'none';

    const nameSpan = document.createElement('span');
    nameSpan.textContent = fileName;
    tab.appendChild(nameSpan);

    // ❌ bouton fermeture
    const closeBtn = document.createElement('span');
    closeBtn.innerHTML = '&times;';
    closeBtn.style.marginLeft = '8px';
    closeBtn.style.color = 'red';
    closeBtn.style.cursor = 'pointer';
    closeBtn.onclick = (e) => {
        e.stopPropagation();
        closeTab(path);
    };
    tab.appendChild(closeBtn);

    // clic sur le nom → activer l’onglet
    tab.onclick = () => setActiveTab(path);

    tabsBar.appendChild(tab);
    return tab;
}
function setActiveTab(path) {
    const ed = initEditor();
    activeFile = path;
    document.getElementById('file-path').innerText = path;

    // Changer la session active
    ed.setSession(openFiles[path].session);

    // Changer le style visuel des onglets
    for (const [p, { tabEl }] of Object.entries(openFiles)) {
        tabEl.style.background = p === path ? '#ccc' : '#eee';
        tabEl.style.fontWeight = p === path ? 'bold' : 'normal';
    }
}
function closeTab(path) {
    const { tabEl } = openFiles[path];
    tabEl.remove();
    delete openFiles[path];

    // Si on ferme l’onglet actif → passer à un autre
    if (activeFile === path) {
        const remaining = Object.keys(openFiles);
        if (remaining.length > 0) setActiveTab(remaining[0]);
        else {
            activeFile = null;
            editor.setValue('');
            document.getElementById('file-path').innerText = '';
        }
    }
}

/**
 * 📂 Ouvrir un fichier avec récupération du .tmp si existant
 */
/*async function openFile(path) {
    try {
        const res = await fetch(`{{ route("file.manager.open") }}?path=${encodeURIComponent(path)}`);
        const data = await res.json();
        if (!data.content) return alert('Impossible de charger le fichier');

        const ed = initEditor();
        ed.session.setMode(getModeByExtension(path));
        ed.setValue(data.content.replace(/\uFEFF/g, '').replace(/\r\n/g, '\n'), -1);
        ed.clearSelection();
        ed.focus();
        currentPath = path;
        document.getElementById('file-path').innerText = path;

        if (data.fromTemp) {
            alert(`⚠️ Un fichier temporaire a été récupéré : ${path}`);
        }

        startAutoSave();

    } catch (err) {
        console.error(err);
        alert("Erreur lors de l'ouverture du fichier");
    }
}*/
async function openFile(path) {
    // Si déjà ouvert → juste activer l'onglet
    if (openFiles[path]) {
        setActiveTab(path);
        return;
    }

    try {
        const res = await fetch(`{{ route("file.manager.open") }}?path=${encodeURIComponent(path)}`);
        const data = await res.json();
        if (!data.content) return alert('Impossible de charger le fichier');

        const ed = initEditor();
        const mode = getModeByExtension(path);

        // 🔹 Crée une session Ace distincte pour ce fichier
        const session = ace.createEditSession(data.content, mode);
        session.setTabSize(4);
        session.setUseSoftTabs(true);

        openFiles[path] = { session, tabEl: createTab(path) };
        setActiveTab(path);

        if (data.fromTemp) alert(`⚠️ Fichier temporaire récupéré : ${path}`);
    } catch (err) {
        console.error(err);
        alert("Erreur lors de l'ouverture du fichier");
    }
}


/**
 * 🔁 Auto-sauvegarde toutes les 20 secondes (.tmp)
 */
function startAutoSave() {
    if (autoSaveInterval) clearInterval(autoSaveInterval);

    
    let lastAutoSavedContent = ''; // Stocke le contenu au dernier auto-save

    // 🔁 Sauvegarde automatique toutes les 20 secondes
    autoSaveInterval = setInterval(async () => {
        if (currentPath) {
            const currentContent = editor.getValue();
    
            // On sauvegarde uniquement si le contenu a changé
            if (currentContent !== lastAutoSavedContent) {
                console.log('💾 Auto-save triggered for', currentPath);
    
                const res = await fetch('{{ route("file.manager.autoSave") }}', {
                    method: 'POST',
                    headers: { 
                        'Content-Type': 'application/json', 
                        'X-CSRF-TOKEN': '{{ csrf_token() }}' 
                    },
                    body: JSON.stringify({ path: currentPath, content: currentContent, temp: true }) // temp=true pour créer fichier tampon
                });
    
                const data = await res.json();
                if (data.success) {
                    console.log('✅ Auto-save OK');
                    lastAutoSavedContent = currentContent; // met à jour la dernière sauvegarde
                }
            }
        }
    }, 20000); // toutes les 20 secondes

}

/**
 * 🌳 Chargement de l’arborescence
 */
async function loadDir(dir = '') {
    const res = await fetch(`{{ route("file.manager.list") }}?dir=${encodeURIComponent(dir)}`);
    const items = await res.json();
    const treeContainer = document.getElementById('file-tree');
    

    items.sort((a, b) => (a.type === b.type) ? a.name.localeCompare(b.name) : (a.type === 'dir' ? -1 : 1));

    const ul = document.createElement('ul');
    ul.classList.add('list-unstyled');

    items.forEach(item => {
        const li = document.createElement('li');
        li.style.paddingLeft = `${(item.depth || 0) * 15}px`;
        li.style.cursor = 'pointer';
        li.oncontextmenu = e => showContextMenu(e, dir, item);

        if (item.type === 'dir') {
            li.innerHTML = `<span class="folder-icon">📁</span> <span class="folder-label">${item.name}</span>`;
            const childrenUl = document.createElement('ul');
            childrenUl.classList.add('list-unstyled', 'ps-2');
            childrenUl.style.display = openedDirs.has(item.path) ? 'block' : 'none';
            li.appendChild(childrenUl);

            const toggleFolder = async (e) => {
                e.stopPropagation();
                if (childrenUl.style.display === 'none') {
                    openedDirs.add(item.path);
                    childrenUl.style.display = 'block';
                    if (childrenUl.children.length === 0) {
                        await loadSubDir(item.path, childrenUl);
                    }
                } else {
                    openedDirs.delete(item.path);
                    childrenUl.style.display = 'none';
                }
            };
            
            // 🔹 Appliquer le même comportement à l’icône ET au label
            li.querySelectorAll('.folder-icon, .folder-label').forEach(el => {
                el.addEventListener('click', toggleFolder);
            });
        } else {
            li.innerHTML = '📄 ' + item.name;
            li.addEventListener('click', () => openFile(item.path));
        }

        ul.appendChild(li);
    });

    if (dir === '') {
        treeContainer.innerHTML = '';
        treeContainer.appendChild(ul);
    }
}

async function loadSubDir(dir, containerUl) {
    const res = await fetch(`{{ route("file.manager.list") }}?dir=${encodeURIComponent(dir)}`);
    const items = await res.json();
    currentPath = dir;
    items.sort((a, b) => (a.type === b.type) ? a.name.localeCompare(b.name) : (a.type === 'dir' ? -1 : 1));

    items.forEach(item => {
        const li = document.createElement('li');
        li.style.paddingLeft = `${(item.depth || 0) * 15}px`;
        li.style.cursor = 'pointer';
        li.oncontextmenu = e => showContextMenu(e, dir, item);

        if (item.type === 'dir') {
            li.innerHTML = `<span class="folder-icon">📁</span> <span class="folder-label">${item.name}</span>`;
            const childrenUl = document.createElement('ul');
            childrenUl.classList.add('list-unstyled', 'ps-2');
            childrenUl.style.display = 'none';
            li.appendChild(childrenUl);
            const toggleFolder = async (e) => {
                e.stopPropagation();
                if (childrenUl.style.display === 'none') {
                    openedDirs.add(item.path);
                    childrenUl.style.display = 'block';
                    if (childrenUl.children.length === 0) {
                        await loadSubDir(item.path, childrenUl);
                    }
                } else {
                    openedDirs.delete(item.path);
                    childrenUl.style.display = 'none';
                }
            };
            
            // 🔹 Appliquer le même comportement à l’icône ET au label
            li.querySelectorAll('.folder-icon, .folder-label').forEach(el => {
                el.addEventListener('click', toggleFolder);
            });
        } else {
            li.innerHTML = '📄 ' + item.name;
            li.addEventListener('click', () => openFile(item.path));
        }

        containerUl.appendChild(li);
    });
}

/**
 * ➕ Nouveau fichier / dossier
 */
async function newItemm(type) {
    const name = prompt(`Nom du nouveau ${type === 'dir' ? 'dossier' : 'fichier'} :`);
    if (!name) return;
    let parentPath = '';

    if (currentPath) {
        if (currentPath.includes('.')) {
            // 👉 suppose que c’est un fichier (car il a une extension)
            parentPath = currentPath.includes('/') ? currentPath.split('/').slice(0, -1).join('/') : '';
        } else {
            // 👉 c’est un dossier, on garde tel quel
            parentPath = currentPath;
        }
    }

    const res = await fetch('{{ route("file.manager.new") }}', {
        method: 'POST',
        headers: { 
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: JSON.stringify({ 
            path: parentPath, 
            name, 
            type 
        })
    });

    const data = await res.json();
    if (data.success) {
        alert('✅ Créé avec succès !');
        loadDir();
    } else {
        alert('❌ Erreur : ' + data.message);
    }
}
async function newItem(type) {
    const name = prompt(`Nom du nouveau ${type === 'dir' ? 'dossier' : 'fichier'} :`);
    if (!name) return;

    let parentPath = '';

    // Si un fichier est actif -> on crée dans son dossier
    if (activeFile) {
        parentPath = activeFile.split('/').slice(0, -1).join('/');
    }
    // Sinon currentPath = dossier actuellement sélectionné dans l’arborescence
    else if (currentPath) {
        parentPath = currentPath;
    }

    const res = await fetch('{{ route("file.manager.new") }}', {
        method: 'POST',
        headers: { 
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: JSON.stringify({ 
            path: parentPath, 
            name, 
            type 
        })
    });

    const data = await res.json();
    if (data.success) {
        alert('✅ Créé avec succès !');
        loadDir();
    } else {
        alert('❌ Erreur : ' + data.message);
    }
}


/**
 * 🗑️ Supprimer / ✏️ Renommer
 */
function showContextMenu(e, dir, item) {
    e.preventDefault();
    removeContextMenu();
    const menu = document.createElement('div');
    menu.id = 'context-menu';
    menu.style.position = 'absolute';
    menu.style.left = e.pageX + 'px';
    menu.style.top = e.pageY + 'px';
    menu.style.background = '#333';
    menu.style.color = 'white';
    menu.style.padding = '5px';
    menu.style.borderRadius = '5px';
    menu.style.zIndex = '1000';
    menu.style.minWidth = '120px';

    const actions = [
        { text: '✏️ Renommer', fn: () => renameItem(dir, item) },
        { text: '🗑️ Supprimer', fn: () => deleteItem(dir, item) }
    ];

    actions.forEach(a => {
        const btn = document.createElement('div');
        btn.textContent = a.text;
        btn.style.cursor = 'pointer';
        btn.style.padding = '2px 5px';
        btn.onmouseenter = () => btn.style.background = '#555';
        btn.onmouseleave = () => btn.style.background = 'transparent';
        btn.onclick = () => { a.fn(); removeContextMenu(); };
        menu.appendChild(btn);
    });

    document.body.appendChild(menu);
}

function removeContextMenu() {
    const old = document.getElementById('context-menu');
    if (old) old.remove();
}
document.body.addEventListener('click', removeContextMenu);

async function deleteItem(dir, item) {
    if (!confirm(`Supprimer ${item.name} ?`)) return;
    const path = (dir ? dir + '/' : '') + item.name;
    const res = await fetch('/file-manager/delete', {
        method: 'DELETE',
        headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': '{{ csrf_token() }}' },
        body: JSON.stringify({ path })
    });
    if ((await res.json()).success) loadDir();
}

async function renameItem(dir, item) {
    const newName = prompt('Nouveau nom :', item.name);
    if (!newName) return;
    const old = (dir ? dir + '/' : '') + item.name;
    const newPath = (dir ? dir + '/' : '') + newName;
    const res = await fetch('/file-manager/rename', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': '{{ csrf_token() }}' },
        body: JSON.stringify({ old, new: newPath })
    });
    if ((await res.json()).success) loadDir();
}

// 🚀 Initialisation
document.addEventListener('DOMContentLoaded', () => {
    initEditor();
    loadDir();
    
    /**
     * 💾 Sauvegarde manuelle + backup
     */
    
    document.getElementById('save-btn').addEventListener('click', async () => {
        if (!activeFile) return alert('Aucun fichier actif');
    
        const content = openFiles[activeFile].session.getValue();
        const res = await fetch('{{ route("file.manager.save") }}', {
            method: 'POST',
            headers: { 
                'Content-Type': 'application/json', 
                'X-CSRF-TOKEN': '{{ csrf_token() }}' 
            },
            body: JSON.stringify({ path: activeFile, content })
        });
        const data = await res.json();
        alert(data.success ? `✅ Sauvegardé : ${activeFile}` : '❌ Erreur lors de la sauvegarde');
    });

    document.getElementById('upload-btn').addEventListener('click', async () => {
        const fileInput = document.getElementById('file-upload');
        if (!fileInput.files.length) return alert('Sélectionnez un fichier à uploader !');
    
        const file = fileInput.files[0];
        const formData = new FormData();
    
        // On envoie le fichier
        formData.append('file', file);
    
        // On précise le dossier cible, par exemple activeFile ou currentPath
        // Si tu veux toujours uploader dans le dossier parent de l’onglet actif :
        const targetDir = activeFile ? activeFile.split('/').slice(0, -1).join('/') : currentPath;
        formData.append('targetDir', targetDir);
    
        try {
            const res = await fetch('{{ route("file.manager.upload") }}', {
                method: 'POST',
                headers: { 'X-CSRF-TOKEN': '{{ csrf_token() }}' },
                body: formData
            });
    
            const data = await res.json();
            if (data.success) {
                alert('✅ Fichier uploadé avec succès !');
                loadDir(); // Recharge l’arborescence
            } else {
                alert('❌ Erreur : ' + data.message);
            }
        } catch (err) {
            console.error(err);
            alert('Erreur lors de l’upload');
        }
    });

});

</script>
@endsection